<?php
require_once 'config.php';

// Check API Key
if (ENABLE_API_KEY_CHECK) {
    $apiKey = $_SERVER['HTTP_X_API_KEY'] ?? $_POST['api_key'] ?? '';
    if ($apiKey !== API_KEY) {
        http_response_code(401);
        die(json_encode(['error' => 'Invalid API key']));
    }
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    die(json_encode(['error' => 'Method not allowed']));
}

// Validate required fields
$place_id = $_POST['place_id'] ?? null;
$map_version = $_POST['map_version'] ?? 1;
$is_checkpoint = isset($_POST['is_checkpoint']) ? (bool)$_POST['is_checkpoint'] : false;
$checkpoint_index = $_POST['checkpoint_index'] ?? null;
$uploader_id = $_POST['uploader_id'] ?? null;
$uploader_name = $_POST['uploader_name'] ?? 'Unknown';
$notes = $_POST['notes'] ?? '';

if (!$place_id) {
    http_response_code(400);
    die(json_encode(['error' => 'place_id is required']));
}

// Handle file upload
if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
    http_response_code(400);
    die(json_encode(['error' => 'No file uploaded or upload error']));
}

$file = $_FILES['file'];
$fileSize = $file['size'];
$tmpName = $file['tmp_name'];
$originalName = basename($file['name']);

// Validate file size
if ($fileSize > MAX_FILE_SIZE) {
    http_response_code(400);
    die(json_encode(['error' => 'File too large (max 100MB)']));
}

// Validate file extension
$ext = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
if (!in_array($ext, ALLOWED_EXTENSIONS)) {
    http_response_code(400);
    die(json_encode(['error' => 'Invalid file type. Only JSON allowed']));
}

// Create directory structure: uploads/paths/{place_id}/v{version}/
$dirPath = UPLOAD_DIR . $place_id . '/v' . $map_version . '/';
if (!file_exists($dirPath)) {
    mkdir($dirPath, 0755, true);
}

// Generate filename
if ($is_checkpoint) {
    $fileName = "checkpoint_{$checkpoint_index}.json";
} else {
    $fileName = "main.json";
}

$filePath = $dirPath . $fileName;
$relativePath = $place_id . '/v' . $map_version . '/' . $fileName;

// Move uploaded file
if (!move_uploaded_file($tmpName, $filePath)) {
    http_response_code(500);
    die(json_encode(['error' => 'Failed to save file']));
}

try {
    $db = getDB();
    
    // Check if path already exists
    $stmt = $db->prepare("
        SELECT id FROM paths 
        WHERE place_id = ? AND map_version = ? AND is_checkpoint = ? AND checkpoint_index <=> ?
    ");
    $stmt->execute([$place_id, $map_version, $is_checkpoint, $checkpoint_index]);
    $existing = $stmt->fetch();
    
    if ($existing) {
        // Update existing
        $stmt = $db->prepare("
            UPDATE paths SET 
                file_path = ?,
                file_size = ?,
                upload_date = NOW(),
                uploader_id = ?,
                uploader_name = ?,
                notes = ?
            WHERE id = ?
        ");
        $stmt->execute([$relativePath, $fileSize, $uploader_id, $uploader_name, $notes, $existing['id']]);
        $pathId = $existing['id'];
    } else {
        // Insert new
        $stmt = $db->prepare("
            INSERT INTO paths (place_id, map_version, is_checkpoint, checkpoint_index, file_path, file_size, uploader_id, uploader_name, notes)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([$place_id, $map_version, $is_checkpoint, $checkpoint_index, $relativePath, $fileSize, $uploader_id, $uploader_name, $notes]);
        $pathId = $db->lastInsertId();
    }
    
    echo json_encode([
        'success' => true,
        'path_id' => $pathId,
        'file_path' => $relativePath,
        'file_size' => $fileSize,
        'message' => 'File uploaded successfully'
    ]);
    
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
}
?>
